<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Autoplaying Logos with Movie Info</title>
    <style>
        body {
            margin: 0; /* Remove margens padrão do corpo */
            height: 100vh; /* Altura total da viewport */
            display: flex;
            justify-content: flex-end; /* Alinha conteúdo à direita */
            align-items: center; /* Centraliza verticalmente */
            background-color: #111; /* Cor de fundo escura */
            overflow: hidden; /* Esconde overflow para evitar scrollbars */
        }

        .backdrop-container {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: -1; /* Garante que o backdrop esteja atrás dos outros elementos */
            overflow: hidden; /* Para evitar overflow do background */
        }

        .backdrop {
            width: 100%;
            height: 100%;
            object-fit: cover;
            filter: blur(8px); /* Aplica o desfoque no fundo */
        }

        .movie-info-poster-container {
            position: relative; /* Relativo para conter o overlay */
            display: flex;
            align-items: center; /* Centraliza verticalmente */
            background-color: transparent; /* Fundo inicial transparente */
            max-width: 60%; /* Largura máxima do container */
            right: 10%;
            box-sizing: border-box; /* Inclui padding no tamanho total */
            overflow: hidden; /* Esconde overflow para evitar scrollbars */
        }

        /* Caixa com opacidade, borda arredondada e blur */
        .info-box {
            background-color: rgba(0, 0, 0, 0.9); /* Fundo preto com opacidade */
            backdrop-filter: blur(20px); /* Aplica o blur à caixa */
            border-radius: 20px; /* Bordas arredondadas */
            padding: 20px;
            display: flex;
            align-items: center; /* Centraliza itens verticalmente */
            flex: 1; /* Permite que a caixa ocupe o espaço disponível */
            margin-right: 20px; /* Espaçamento à direita */
        }

        .poster {
            height: 50px; /* Altura do pôster ajustada */
            border-radius: 9px; /* Arredondamento das bordas do pôster */
            border: none; /* Remove a borda branca */
            box-shadow: none; /* Remove a sombra do pôster */
            object-fit: cover; /* Garante que a imagem cubra a área sem distorção */
            margin-left: 10px; /* Espaçamento à esquerda do pôster */
        }

        .movie-info {
            font-size: 9px; /* Tamanho do texto informativo */
            color: white;
            line-height: 1.1em; /* Altura da linha para espaçamento */
            overflow: hidden; /* Esconde texto que excede a altura máxima */
            text-overflow: ellipsis; /* Indica que o texto foi cortado */
            white-space: normal; /* Permite que o texto quebre em várias linhas */
            text-shadow: 2px 2px 2px rgba(0, 0, 0, 1.8); /* Adiciona a sombra ao texto */
            display: flex;
            flex-direction: column; /* Alinha os elementos em coluna */
            justify-content: center; /* Centraliza verticalmente */
            flex: 1; /* Expande para ocupar espaço disponível */
        }

        .movie-title {
            font-size: 10px; /* Tamanho do título do filme */
            font-weight: bold; /* Destaque para o título */
            margin-bottom: 9px; /* Espaçamento abaixo do título */
        }

        .visible {
            opacity: 1; /* Torna visível */
        }
    </style>
</head>
<body>
    <div class="backdrop-container" id="backdrop-container">
        <img id="backdrop" class="backdrop">
    </div>
    <div class="movie-info-poster-container">
        <div class="info-box">
            <div class="movie-info" id="movie-info"></div>
            <img id="poster" class="poster">
        </div>
    </div>
    <script>
        const apiKey = '6b8e3eaa1a03ebb45642e9531d8a76d2';
        let currentIndex = 0;
        let movieIds = [];

        async function fetchPopularEnglishMovieIds() {
            try {
                const response = await fetch(`https://api.themoviedb.org/3/discover/movie?api_key=${apiKey}&sort_by=popularity.desc&language=pt`);
                if (!response.ok) {
                    throw new Error('Failed to fetch popular English movies');
                }
                const data = await response.json();
                movieIds = data.results.map(movie => movie.id);
            } catch (error) {
                console.error('Error fetching popular English movies:', error);
            }
        }

        async function fetchMovieBackdrop(movieId) {
            try {
                const response = await fetch(`https://api.themoviedb.org/3/movie/${movieId}/images?api_key=${apiKey}`);
                if (!response.ok) {
                    throw new Error('Failed to fetch movie backdrop');
                }
                const data = await response.json();
                const backdropPath = data.backdrops[0].file_path; // Assumindo o primeiro backdrop do array
                return `https://image.tmdb.org/t/p/original${backdropPath}`;
            } catch (error) {
                console.error('Error fetching movie backdrop:', error);
            }
        }

        async function fetchMoviePoster(movieId) {
            try {
                const response = await fetch(`https://api.themoviedb.org/3/movie/${movieId}?api_key=${apiKey}&language=pt`);
                if (!response.ok) {
                    throw new Error('Failed to fetch movie poster');
                }
                const data = await response.json();
                return `https://image.tmdb.org/t/p/w300${data.poster_path}`;
            } catch (error) {
                console.error('Error fetching movie poster:', error);
            }
        }

        async function fetchMovieInfo(movieId) {
            try {
                const response = await fetch(`https://api.themoviedb.org/3/movie/${movieId}?api_key=${apiKey}&language=pt`);
                if (!response.ok) {
                    throw new Error('Failed to fetch movie info');
                }
                const data = await response.json();
                return {
                    title: data.title,
                    releaseDate: data.release_date,
                    overview: data.overview
                };
            } catch (error) {
                console.error('Error fetching movie info:', error);
            }
        }

        function truncateText(text, maxLength) {
            return text.length > maxLength ? text.substring(0, maxLength) + '...' : text;
        }

        async function updateBackdropPosterAndInfo() {
            if (movieIds.length === 0) {
                console.error('No movie IDs available.');
                return;
            }

            const movieId = movieIds[currentIndex];

            try {
                const backdropUrl = await fetchMovieBackdrop(movieId);
                const posterUrl = await fetchMoviePoster(movieId);
                const movieInfo = await fetchMovieInfo(movieId);
                const movieInfoElement = document.getElementById('movie-info');
                const backdrop = document.getElementById('backdrop');
                const poster = document.getElementById('poster');

                backdrop.classList.add('visible');
                movieInfoElement.classList.add('visible');
                poster.classList.add('visible');

                setTimeout(() => {
                    if (backdropUrl) {
                        backdrop.src = backdropUrl;
                    }

                    if (posterUrl) {
                        poster.src = posterUrl;
                    }

                    if (movieInfo) {
                        const overview = truncateText(movieInfo.overview, 200); // Limita a 200 caracteres
                        movieInfoElement.innerHTML = `
                            <div class="movie-title">${movieInfo.title} (${movieInfo.releaseDate.substring(0, 4)})</div>
                            ${overview}
                        `;
                    }

                }, 1000);

                currentIndex = (currentIndex + 1) % movieIds.length;
            } catch (error) {
                console.error('Error updating backdrop, poster, and movie info:', error);
            }
        }

        fetchPopularEnglishMovieIds().then(() => {
            setInterval(updateBackdropPosterAndInfo, 10000); // Muda o backdrop, pôster e info a cada 10 segundos
            updateBackdropPosterAndInfo(); // Atualização inicial
        });
    </script>
</body>
</html>
